# Copyright 2002-2025, OpenNebula Project, OpenNebula Systems
#
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
#
# http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.

from __future__ import annotations

import os
from typing import ClassVar

import numpy as np
import pandas as pd
from prophet import Prophet

from ..core.tsnumpy.index.time import TimeIndex
from ..core.tsnumpy.timeseries import Timeseries
from .base_prediction_model import BasePredictionModel
from .model_config import ModelConfig


class ProphetPredictionModel(BasePredictionModel):
    """
    Implement an prediction model based on PROPHET [1]

    Attributes
    ----------
    model_config : ModelConfig
        The configuration of the model.


    References
    ----------
    [1] Taylor, S. J., & Letham, B. (2018).
        Forecasting at scale. The American Statistician, 72(1), 37-45.
    """

    __SUPPORTS_CI__: ClassVar[bool] = True
    prophet: Prophet | None = None

    @staticmethod
    def ts_to_dataframe(metric: Timeseries) -> pd.DataFrame:
        if not metric.is_univariate:
            raise ValueError(
                "The current Prophet-based prediction model only supports "
                "univariate timeseries."
            )
        load = {"ds": metric.time_index, "y": metric.values.flatten()}
        df = pd.DataFrame(load, index=metric.time_index)
        df["ds"] = df["ds"].dt.tz_localize(None)
        return df

    def fit(
        self,
        metric: Timeseries,
    ) -> ProphetPredictionModel:
        if not isinstance(metric, Timeseries):
            raise TypeError("The metric must be a Timeseries object.")
        if not metric.is_univariate:
            raise ValueError(
                "The current Prophet-based prediction model only supports "
                "univariate timeseries."
            )
        input_data = self.ts_to_dataframe(metric)
        self.prophet = Prophet()
        # NOTE: we need to define expected potential seasonality patterns
        # for the Prophet model to work correctly with the data.
        # self.prophet.add_seasonality("secondly", period=1/86400, fourier_order=3)
        self.prophet.add_seasonality(
            "minutely", period=1 / 1440, fourier_order=1
        )
        self.prophet.add_seasonality("hourly", period=1 / 24, fourier_order=3)
        self.prophet.add_seasonality(name="daily", period=1, fourier_order=3)
        self.prophet.add_seasonality(name="weekly", period=7, fourier_order=3)
        self.prophet.fit(input_data)
        return self

    def predict(
        self,
        metric: Timeseries,
        horizon: int | None = None,
        forecast_index: TimeIndex | None = None,
    ) -> Timeseries:
        """
        Generate predictions using the Prophet model.

        Parameters
        ----------
        metric : Timeseries
            The metric data (univariate or multivariate) used for
            fitting the Prophet model and making predictions.
        horizon : int or None
            The number of time steps to predict. If None, the
            prediction horizon is determined by the model.
        forecast_index : TimeIndex or None
            The time index for the forecast. If None, the time index
            is generated based on the last time step of the metric.
        Returns
        -------
        Timeseries
            A Timeseries object containing the predictions generated by the
            Prophet model.
        """
        forecast_index = self._forecast_time_index(
            metric, horizon, forecast_index
        )
        multivariate = []
        for mattr, entity_uid, ts in metric.iter_over_variates():
            self.fit(ts)
            foreacst_df = pd.DataFrame({"ds": forecast_index.values})
            foreacst_df["ds"] = foreacst_df["ds"].dt.tz_localize(None)
            result = self.prophet.predict(foreacst_df)
            forecast_data = result["yhat"].values

            forecast_ts = Timeseries(
                time_idx=forecast_index.values,
                metric_idx=np.array([mattr]),
                entity_uid_idx=np.array([entity_uid]),
                data=np.array(forecast_data).reshape(-1, 1, 1),
            )
            multivariate.append(forecast_ts)
        return Timeseries.merge(*multivariate)

    @classmethod
    def load(
        cls,
        model_config: ModelConfig,
        checkpoint: str | os.PathLike | None = None,
    ) -> ProphetPredictionModel:
        return cls(model_config)
