/*******************************************************************************
 * Copyright 2002-2025, OpenNebula Project, OpenNebula Systems
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 ******************************************************************************/
package org.opennebula.client.vmgroup;


import org.opennebula.client.Client;
import org.opennebula.client.OneResponse;
import org.opennebula.client.PoolElement;
import org.w3c.dom.Node;

/**
 * This class represents an OpenNebula vmgroup.
 * It also offers static XML-RPC call wrappers.
 */
public class VMGroup extends PoolElement{

    private static final String METHOD_PREFIX   = "vmgroup.";
    private static final String ALLOCATE        = METHOD_PREFIX + "allocate";
    private static final String INFO            = METHOD_PREFIX + "info";
    private static final String UPDATE          = METHOD_PREFIX + "update";
    private static final String DELETE          = METHOD_PREFIX + "delete";
    private static final String CHOWN           = METHOD_PREFIX + "chown";
    private static final String CHMOD           = METHOD_PREFIX + "chmod";
    private static final String RENAME          = METHOD_PREFIX + "rename";
    private static final String LOCK            = METHOD_PREFIX + "lock";
    private static final String UNLOCK          = METHOD_PREFIX + "unlock";
    private static final String ROLEADD         = METHOD_PREFIX + "roleadd";
    private static final String ROLEDELETE      = METHOD_PREFIX + "roledelete";
    private static final String ROLEUPDATE      = METHOD_PREFIX + "roleupdate";

    /**
     * Creates a new vmgroup representation.
     *
     * @param id The vmgroup id.
     * @param client XML-RPC Client.
     */
    public VMGroup(int id, Client client)
    {
        super(id, client);
    }

    /**
     * @see PoolElement
     */
    protected VMGroup(Node xmlElement, Client client)
    {
        super(xmlElement, client);
    }

    // =================================
    // Static XML-RPC methods
    // =================================

    /**
     * Allocates a new vmgroup in OpenNebula.
     *
     * @param client XML-RPC Client.
     * @param description A string containing the template
     * of the vmgroup.
     *
     * @return If successful the message contains the associated
     * id generated for this vmgroup.
     */
    public static OneResponse allocate(
            Client  client,
            String  description)
    {
        return client.call(ALLOCATE, description);
    }

    /**
     * Retrieves the information of the given vmgroup
     *
     * @param client XML-RPC Client.
     * @param id the id for the vmgroup to
     * retrieve the information from.
     * @return If successful the message contains the string
     * with the information returned by OpenNebula.
     */
    public static OneResponse info(Client client, int id)
    {
        return client.call(INFO, id);
    }

    /**
     * Retrieves the information of the given VMGroup.
     *
     * @param client XML-RPC Client.
     * @param id The VMGroup id for the VMGroup to retrieve the information from
     * @param decrypt If true decrypt sensitive attributes
     * @return If successful the message contains the string
     * with the information returned by OpenNebula.
     */
    public static OneResponse info(Client client, int id, boolean decrypt)
    {
        return client.call(INFO, id, decrypt);
    }

    /**
     * Deletes a vmgroup from OpenNebula.
     *
     * @param client XML-RPC Client.
     * @param id The id of the target vmgroup.
     * @return A encapsulated response.
     */
    public static OneResponse delete(Client client, int id)
    {
        return client.call(DELETE, id);
    }

    /**
     * Changes the owner/group
     *
     * @param client XML-RPC Client.
     * @param id The id of the target vmgroup.
     * @param uid The new owner user ID. Set it to -1 to leave the current one.
     * @param gid The new group ID. Set it to -1 to leave the current one.
     * @return If an error occurs the error message contains the reason.
     */
    public static OneResponse chown(Client client, int id, int uid, int gid)
    {
        return client.call(CHOWN, id, uid, gid);
    }

    /**
     * Changes the vmgroup permissions
     *
     * @param client XML-RPC Client.
     * @param id The id of the target vmgroup.
     * @param owner_u 1 to allow, 0 deny, -1 do not change
     * @param owner_m 1 to allow, 0 deny, -1 do not change
     * @param owner_a 1 to allow, 0 deny, -1 do not change
     * @param group_u 1 to allow, 0 deny, -1 do not change
     * @param group_m 1 to allow, 0 deny, -1 do not change
     * @param group_a 1 to allow, 0 deny, -1 do not change
     * @param other_u 1 to allow, 0 deny, -1 do not change
     * @param other_m 1 to allow, 0 deny, -1 do not change
     * @param other_a 1 to allow, 0 deny, -1 do not change
     * @return If an error occurs the error message contains the reason.
     */
    public static OneResponse chmod(Client client, int id,
                                    int owner_u, int owner_m, int owner_a,
                                    int group_u, int group_m, int group_a,
                                    int other_u, int other_m, int other_a)
    {
        return chmod(client, CHMOD, id,
                owner_u, owner_m, owner_a,
                group_u, group_m, group_a,
                other_u, other_m, other_a);
    }

    /**
     * Changes the permissions
     *
     * @param client XML-RPC Client.
     * @param id The id of the target object.
     * @param octet Permissions octed , e.g. 640
     * @return If an error occurs the error message contains the reason.
     */
    public static OneResponse chmod(Client client, int id, String octet)
    {
        return chmod(client, CHMOD, id, octet);
    }

    /**
     * Changes the permissions
     *
     * @param client XML-RPC Client.
     * @param id The id of the target object.
     * @param octet Permissions octed , e.g. 640
     * @return If an error occurs the error message contains the reason.
     */
    public static OneResponse chmod(Client client, int id, int octet)
    {
        return chmod(client, CHMOD, id, octet);
    }

    /**
     * Replaces the vmgroup template contents.
     *
     * @param client XML-RPC Client.
     * @param id The id of the target vmgroup we want to modify.
     * @param new_template New template contents.
     * @param append True to append new attributes instead of replace the whole template
     * @return If successful the message contains the vmgroup id.
     */
    public static OneResponse update(Client client, int id, String new_template,
        boolean append)
    {
        return client.call(UPDATE, id, new_template, append ? 1 : 0);
    }

    /**
     * Renames this vmgroup
     *
     * @param client XML-RPC Client.
     * @param id The id of the target vmgroup.
     * @param name New name for the vmgroup.
     * @return If an error occurs the error message contains the reason.
     */
    public static OneResponse rename(Client client, int id, String name)
    {
        return client.call(RENAME, id, name);
    }

    /**
     * lock this vmgroup
     *
     * @param client XML-RPC Client.
     * @param id The id of the target vmgroup.
     * @param level Lock level.
     * @return If an error occurs the error message contains the reason.
     */
    public static OneResponse lock(Client client, int id, int level)
    {
        return client.call(LOCK, id, level);
    }

    /**
     * Unlock this vmgroup
     *
     * @param client XML-RPC Client.
     * @param id The id of the target vmgroup.
     * @return If an error occurs the error message contains the reason.
     */
    public static OneResponse unlock(Client client, int id)
    {
        return client.call(UNLOCK, id);
    }

    /**
     * Add VM Group role
     *
     * @param client XML-RPC Client.
     * @param id The id of the target VM Group.
     * @param roleTemplate New template content.
     * @return If an error occurs the error message contains the reason.
     */
    public static OneResponse roleAdd(Client client, int id, String roleTemplate)
    {
        return client.call(ROLEADD, id, roleTemplate);
    }

    /**
     * Delete VM Group role
     *
     * @param client XML-RPC Client.
     * @param id The id of the target VM Group.
     * @param roleID The id of the target role.
     * @return If an error occurs the error message contains the reason.
     */
    public static OneResponse roleDelete(Client client, int id, int roleID)
    {
        return client.call(ROLEDELETE, id, roleID);
    }

    /**
     * Unlock this vmgroup
     *
     * @param client XML-RPC Client.
     * @param id The id of the target VM Group.
     * @param roleID The id of the target role.
     * @param roleTemplate Updated template content.
     * @return If an error occurs the error message contains the reason.
     */
    public static OneResponse roleUpdate(Client client, int id, int roleID, String roleTemplate)
    {
        return client.call(ROLEUPDATE, id, roleID, roleTemplate);
    }

    // =================================
    // Instanced object XML-RPC methods
    // =================================

    /**
     * Loads the xml representation of the vmgroup.
     * The info is also stored internally.
     *
     * @see VMGroup#info(Client, int)
     */
    public OneResponse info()
    {
        OneResponse response = info(client, id);
        super.processInfo(response);
        return response;
    }

    /**
     * Deletes the vmgroup from OpenNebula.
     *
     * @return A encapsulated response.
     */
    public OneResponse delete()
    {
        return delete(client, id);
    }

    /**
     * Changes the owner/group
     *
     * @param uid The new owner user ID. Set it to -1 to leave the current one.
     * @param gid The new group ID. Set it to -1 to leave the current one.
     * @return If an error occurs the error message contains the reason.
     */
    public OneResponse chown(int uid, int gid)
    {
        return chown(client, id, uid, gid);
    }

    /**
     * Changes the owner
     *
     * @param uid The new owner user ID.
     * @return If an error occurs the error message contains the reason.
     */
    public OneResponse chown(int uid)
    {
        return chown(uid, -1);
    }

    /**
     * Changes the group
     *
     * @param gid The new group ID.
     * @return If an error occurs the error message contains the reason.
     */
    public OneResponse chgrp(int gid)
    {
        return chown(-1, gid);
    }

    /**
     * Changes the vmgroup permissions
     *
     * @param owner_u 1 to allow, 0 deny, -1 do not change
     * @param owner_m 1 to allow, 0 deny, -1 do not change
     * @param owner_a 1 to allow, 0 deny, -1 do not change
     * @param group_u 1 to allow, 0 deny, -1 do not change
     * @param group_m 1 to allow, 0 deny, -1 do not change
     * @param group_a 1 to allow, 0 deny, -1 do not change
     * @param other_u 1 to allow, 0 deny, -1 do not change
     * @param other_m 1 to allow, 0 deny, -1 do not change
     * @param other_a 1 to allow, 0 deny, -1 do not change
     * @return If an error occurs the error message contains the reason.
     */
    public OneResponse chmod(int owner_u, int owner_m, int owner_a,
                             int group_u, int group_m, int group_a,
                             int other_u, int other_m, int other_a)
    {
        return chmod(client, id,
                    owner_u, owner_m, owner_a,
                    group_u, group_m, group_a,
                    other_u, other_m, other_a);
    }

    /**
     * Changes the permissions
     *
     * @param octet Permissions octed , e.g. 640
     * @return If an error occurs the error message contains the reason.
     */
    public OneResponse chmod(String octet)
    {
        return chmod(client, id, octet);
    }

    /**
     * Changes the permissions
     *
     * @param octet Permissions octed , e.g. 640
     * @return If an error occurs the error message contains the reason.
     */
    public OneResponse chmod(int octet)
    {
        return chmod(client, id, octet);
    }

    /**
     * Replaces the vmgroup template contents.
     *
     * @param new_template New template contents.
     * @return If successful the message contains the vmgroup id.
     */
    public OneResponse update(String new_template)
    {
        return update(new_template, false);
    }

    /**
     * Replaces the vmgroup template contents.
     *
     * @param new_template New template contents.
     * @param append True to append new attributes instead of replace the whole template
     * @return If successful the message contains the vmgroup id.
     */
    public OneResponse update(String new_template, boolean append)
    {
        return update(client, id, new_template, append);
    }

    /**
     * Renames this vmgroup
     *
     * @param name New name for the vmgroup.
     * @return If an error occurs the error message contains the reason.
     */
    public OneResponse rename(String name)
    {
        return rename(client, id, name);
    }

    /**
     * Lock this vmgroup
     *
     * @param level Lock level.
     * @return If an error occurs the error message contains the reason.
     */
    public OneResponse lock(int level)
    {
        return lock(client, id, level);
    }

    /**
     * Unlock this vmgroup
     *
     * @return If an error occurs the error message contains the reason.
     */
    public OneResponse unlock()
    {
        return unlock(client, id);
    }

    /**
     * Add VM Group role
     *
     * @param roleTemplate Updated template content.
     * @return If an error occurs the error message contains the reason.
     */
    public OneResponse roleAdd(String roleTemplate)
    {
        return roleAdd(client, id, roleTemplate);
    }

    /**
     * Delete VM Group role
     *
     * @param roleID The id of the target role.
     * @return If an error occurs the error message contains the reason.
     */
    public OneResponse roleDelete(int roleID)
    {
        return roleDelete(client, id, roleID);
    }

    /**
     * Update VM Group role
     *
     * @param roleID The id of the target role.
     * @param roleTemplate Updated template content.
     * @return If an error occurs the error message contains the reason.
     */
    public OneResponse roleUpdate(int roleID, String roleTemplate)
    {
        return roleUpdate(client, id, roleID, roleTemplate);
    }

    // =================================
    // Helpers
    // =================================

}
