/* -------------------------------------------------------------------------- */
/* Copyright 2002-2025, OpenNebula Project, OpenNebula Systems                */
/*                                                                            */
/* Licensed under the Apache License, Version 2.0 (the "License"); you may    */
/* not use this file except in compliance with the License. You may obtain    */
/* a copy of the License at                                                   */
/*                                                                            */
/* http://www.apache.org/licenses/LICENSE-2.0                                 */
/*                                                                            */
/* Unless required by applicable law or agreed to in writing, software        */
/* distributed under the License is distributed on an "AS IS" BASIS,          */
/* WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.   */
/* See the License for the specific language governing permissions and        */
/* limitations under the License.                                             */
/* -------------------------------------------------------------------------- */

#ifndef DOCUMENT_H_
#define DOCUMENT_H_

#include "PoolObjectSQL.h"
#include "DocumentTemplate.h"

/* -------------------------------------------------------------------------- */
/* -------------------------------------------------------------------------- */

/**
 *  The Document class.
 */
class Document : public PoolObjectSQL
{
public:

    virtual ~Document() = default;

    /**
     * Function to print the Document object into a string in XML format
     *  @param xml the resulting XML string
     *  @return a reference to the generated string
     */
    std::string& to_xml(std::string& xml) const override;

    // ------------------------------------------------------------------------
    // Template Contents
    // ------------------------------------------------------------------------

    /**
     *  Factory method for document templates
     */
    std::unique_ptr<Template> get_new_template() const override
    {
        return std::make_unique<DocumentTemplate>();
    }

    /**
     *  Returns a copy of the Template
     *    @return A copy of the VirtualMachineTemplate
     */
    std::unique_ptr<Template> clone_template() const
    {
        return std::make_unique<DocumentTemplate>(*obj_template);
    };

    /**
     * Returns the document type
     *
     * @return the document type
     */
    int get_document_type() const
    {
        return type;
    };


private:
    // -------------------------------------------------------------------------
    // Friends
    // -------------------------------------------------------------------------

    friend class DocumentPool;

    // -------------------------------------------------------------------------
    // Document Attributes
    // -------------------------------------------------------------------------

    /**
     *  Document type, to implement generic objects.
     */
    int         type;

    // *************************************************************************
    // DataBase implementation (Private)
    // *************************************************************************

    /**
     *  Execute an INSERT or REPLACE Sql query.
     *    @param db The SQL DB
     *    @param replace Execute an INSERT or a REPLACE
     *    @param error_str Returns the error reason, if any
     *    @return 0 one success
     */
    int insert_replace(SqlDB *db, bool replace, std::string& error_str);

    /**
     *  Bootstraps the database table(s) associated to the Document
     *    @return 0 on success
     */
    static int bootstrap(SqlDB * db);

    /**
     *  Rebuilds the object from an xml formatted string
     *    @param xml_str The xml-formatted string
     *
     *    @return 0 on success, -1 otherwise
     */
    int from_xml(const std::string &xml_str) override;

protected:

    // *************************************************************************
    // Constructor
    // *************************************************************************
    Document(   int id,
                int uid,
                int gid,
                const std::string& uname,
                const std::string& gname,
                int umask,
                int type,
                std::unique_ptr<Template> _template_contents);

    // *************************************************************************
    // DataBase implementation
    // *************************************************************************

    /**
     *  Writes the Document in the database.
     *    @param db pointer to the db
     *    @param error_str Returns the error reason, if any
     *    @return 0 on success
     */
    int insert(SqlDB *db, std::string& error_str) override;

    /**
     *  Writes/updates the Document data fields in the database.
     *    @param db pointer to the db
     *    @return 0 on success
     */
    int update(SqlDB *db) override
    {
        std::string err;
        return insert_replace(db, true, err);
    };
};

#endif /*DOCUMENT_H_*/
