# Copyright 2002-2025, OpenNebula Project, OpenNebula Systems
#
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
#
# http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.

from __future__ import annotations

import warnings
from typing import ClassVar

import numpy as np

from ..core.time import Instant, Period
from ..core.tsnumpy.timeseries import Timeseries
from .base_prediction_model import BasePredictionModel
from .model_config import ModelConfig
from .utils import get_class


class ArimaPredictionModel(BasePredictionModel):
    """
    Implement an ARIMA prediction model for time series data.

    The ARIMA model is fit on-the-fly during each call to the
    `predict` method.

    Attributes
    ----------
    model_config : ModelConfig
        The configuration of the model.
    model : ArimaModel
        The ARIMA model used, initialized to None until predict.
    __SUPPORTS_CI__ : bool
        Whether the model supports confidence intervals.
    """

    __SUPPORTS_CI__: ClassVar[bool] = False

    def fit(
        self,
        metric: Timeseries,
    ) -> ArimaPredictionModel:
        """
        Do not train the ARIMA model as it will be fitted in the
        `predict` method.

        Parameters
        ----------
        metric : Timeseries
            The metric used to train the model (univariate or
            multivariate).

        Returns
        -------
        ArimaPredictionModel
            The instance of the ArimaPredictionModel, without
            performing any fitting as it's done on-the-fly in
            `predict`.
        """
        return self

    def predict(self, metric: Timeseries, horizon: int = 1) -> Timeseries:
        """
        Generate predictions using the ARIMA model.

        The ARIMA model is fit on-the-fly to the provided metric.

        Parameters
        ----------
        metric : Timeseries
            The metric data (univariate or multivariate) used for
            fitting the ARIMA model and making predictions.
        horizon : int
            The number of steps into the future to predict (default
            is 1).

        Returns
        -------
        Timeseries
            A Timeseries object containing the predictions generated by the
            ARIMA model.
        """
        model_class = get_class(self.model_config.model_class)
        predictions = []

        last_ts = metric.time_index[-1]

        for name in metric.names:
            act_metric = metric[name]
            metric_data = act_metric.to_array()

            if len(metric_data) == 1:
                warnings.warn(
                    "Only one historical step is used. Returning the "
                    " latest value for the prediction."
                )
                last_value = metric_data[-1]
                forecast = np.repeat(last_value, horizon)
            else:
                model = model_class(
                    metric_data, **self.model_config.hyper_params
                )
                fitted_model = model.fit()

                forecast = fitted_model.forecast(steps=horizon)

            if horizon == 1:
                prediction_time_index = [
                    Instant(last_ts + metric.frequency).value
                ]
            else:
                prediction_time_index = Period(
                    slice(
                        last_ts + metric.frequency,
                        last_ts + horizon * metric.frequency,
                        metric.frequency,
                    )
                ).values

            predicted_metric = Timeseries(
                prediction_time_index, {name: forecast}
            )
            list_metric_predictions.append(predicted_metric)

        return Timeseries.multivariate(list_metric_predictions)

    @classmethod
    def load(
        cls, model_config: ModelConfig, checkpoint: io.BytesIO | None = None
    ) -> ArimaPredictionModel:
        """
        Load an ARIMA model with the specified model configuration.

        This method does not utilize checkpoints as ARIMA is fit
        on-the-fly during prediction.

        Parameters
        ----------
        model_config : ModelConfig
            The model configuration used to load the ARIMA model.
        checkpoint : str or os.PathLike or None
            Ignored, as ARIMA does not require checkpoint-based loading
            (default is None).

        Returns
        -------
        ArimaPredictionModel
            An instance of the ArimaPredictionModel with the loaded
            configuration.
        """
        return cls(model_config)
