#!/bin/bash

# -------------------------------------------------------------------------- #
# Copyright 2002-2025, OpenNebula Project, OpenNebula Systems                #
#                                                                            #
# Licensed under the Apache License, Version 2.0 (the "License"); you may    #
# not use this file except in compliance with the License. You may obtain    #
# a copy of the License at                                                   #
#                                                                            #
# http://www.apache.org/licenses/LICENSE-2.0                                 #
#                                                                            #
# Unless required by applicable law or agreed to in writing, software        #
# distributed under the License is distributed on an "AS IS" BASIS,          #
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.   #
# See the License for the specific language governing permissions and        #
# limitations under the License.                                             #
#--------------------------------------------------------------------------- #

set -e

function is_mounted {
    grep -qs "$1" /proc/mounts
}

function clean {
    if is_mounted "${tmp_dir}/${id}"; then
        umount "${tmp_dir}/${id}"
    fi
}

#-------------------------------------------------------------------------------
# Configuration attributes and parameters
#-------------------------------------------------------------------------------

USER=oneadmin
GROUP=oneadmin

DRIVER_PATH=$(dirname $0)

tmp_dir="$1"
id="$2"
extension="$3"
terminal="$4"

commands=$(cat /dev/stdin)

case "$extension" in
    "tar.xz")
        untar_options="xvJpf"
        ;;
    "tar.gz")
        untar_options="xvzpf"
        ;;
esac

#-------------------------------------------------------------------------------
# Mount container disk image and untar rootfs contents to it
#-------------------------------------------------------------------------------

trap clean EXIT

# try first to mount with the fuse2fs command and if that fails fallback to the
# regular mount
# NOTE: fuse2fs returns zero even when actual mount fails
_fuse_failed=''
if ! fuse2fs -o nosuid "${tmp_dir}/${id}.raw" "${tmp_dir}/${id}" >/dev/null 2>&1 ; then
    _fuse_failed=yes
fi

if [ "$_fuse_failed" = "yes" ] || ! is_mounted "${tmp_dir}/${id}" ; then
    mount -o nosuid "${tmp_dir}/${id}.raw" "${tmp_dir}/${id}"
fi

chown $USER:$GROUP "${tmp_dir}/${id}"
tar $untar_options "${tmp_dir}/${id}.${extension}" -C "${tmp_dir}/${id}" > /dev/null 2>&1

sync

#-------------------------------------------------------------------------------
# Execute context script to install OpenNebula contextualization packages
#-------------------------------------------------------------------------------
cat << EOF | chroot "${tmp_dir}/${id}" $terminal
$commands
echo "#Generated by OpenNebula" > /etc/resolv.conf
rm -f /etc/ssh/ssh_host_* > /dev/null 2>&1
usermod -p '*' root > /dev/null 2>&1
EOF
sync

exit 0
