/* ------------------------------------------------------------------------- *
 * Copyright 2002-2025, OpenNebula Project, OpenNebula Systems               *
 *                                                                           *
 * Licensed under the Apache License, Version 2.0 (the "License"); you may   *
 * not use this file except in compliance with the License. You may obtain   *
 * a copy of the License at                                                  *
 *                                                                           *
 * http://www.apache.org/licenses/LICENSE-2.0                                *
 *                                                                           *
 * Unless required by applicable law or agreed to in writing, software       *
 * distributed under the License is distributed on an "AS IS" BASIS,         *
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.  *
 * See the License for the specific language governing permissions and       *
 * limitations under the License.                                            *
 * ------------------------------------------------------------------------- */
import { HostCard } from '@modules/components/Cards'
import { HostAPI, oneApi } from '@FeaturesModule'
import { jsonToXml } from '@ModelsModule'
import PropTypes from 'prop-types'
import { memo, useCallback, useMemo } from 'react'

const Row = memo(
  ({
    original,
    value,
    onClickLabel,
    zone,
    headerList,
    rowDataCy,
    isSelected,
    toggleRowSelected,
    ...props
  }) => {
    const [update] = HostAPI.useUpdateHostMutation()

    const {
      data: hosts,
      error,
      isLoading,
    } = oneApi.endpoints.getHosts.useQueryState({ zone })

    const host = useMemo(
      () => hosts?.find((h) => +h.ID === +original.ID) ?? original,
      [hosts, original]
    )

    const memoHost = useMemo(
      () => host ?? original,
      [host, original, update, isLoading, error, hosts]
    )

    const handleDeleteLabel = useCallback(
      (label) => {
        const currentLabels = memoHost?.TEMPLATE?.LABELS.split(',')
        const newLabels = currentLabels.filter((l) => l !== label).join(',')
        const newHostTemplate = { ...memoHost.TEMPLATE, LABELS: newLabels }
        const templateXml = jsonToXml(newHostTemplate)

        update({ id: original.ID, template: templateXml, replace: 0 })
      },
      [memoHost.TEMPLATE?.LABELS, update]
    )

    return (
      <HostCard
        host={memoHost}
        rootProps={props}
        onClickLabel={onClickLabel}
        onDeleteLabel={handleDeleteLabel}
      />
    )
  },
  (prev, next) => prev.className === next.className
)

Row.propTypes = {
  original: PropTypes.object,
  value: PropTypes.object,
  isSelected: PropTypes.bool,
  className: PropTypes.string,
  handleClick: PropTypes.func,
  onClickLabel: PropTypes.func,
  zone: PropTypes.oneOfType([PropTypes.string, PropTypes.number]),
  headerList: PropTypes.oneOfType([PropTypes.array, PropTypes.bool]),
  rowDataCy: PropTypes.string,
  toggleRowSelected: PropTypes.func,
}

Row.displayName = 'HostRow'

export default Row
